import 'package:admin_desktop/app_constants.dart';
import 'package:admin_desktop/infrastructure/services/extensions.dart';
import 'package:admin_desktop/infrastructure/services/time_service.dart';
import 'package:admin_desktop/presentation/components/dialogs/delete_dialog.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/catalogue/widgets/service/widgets/service_pop_up.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/service_master/add_service_page.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/service_master/update_service_page.dart';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:pull_to_refresh/pull_to_refresh.dart';
import 'package:remixicon_updated/remixicon_updated.dart';
import '../../../../../../../../domain/models/data/service_data.dart';
import '../../../../../../../../infrastructure/services/app_helpers.dart';
import '../../../../../../../../infrastructure/services/tr_keys.dart';
import 'package:admin_desktop/application/add_service/add_service_master_provider.dart';
import '../../../../../infrastructure/services/local_storage.dart';
import 'package:admin_desktop/presentation/components/buttons/custom_refresher.dart';
import 'package:admin_desktop/presentation/components/buttons/has_more_button.dart';
import 'package:admin_desktop/presentation/components/custom_checkbox.dart';
import 'package:admin_desktop/presentation/components/login_button.dart';
import 'package:admin_desktop/presentation/styles/style.dart';
import '../catalogue/widgets/service/riverpod/service/service_provider.dart';
import '../catalogue/widgets/service_master/riverpod/service_master/service_master_provider.dart';

class ServiceMasterTablePage extends ConsumerStatefulWidget {
  const ServiceMasterTablePage({super.key});

  @override
  ConsumerState<ServiceMasterTablePage> createState() =>
      _ServiceMasterTablePageState();
}

class _ServiceMasterTablePageState
    extends ConsumerState<ServiceMasterTablePage> {
  late RefreshController refreshController;

  @override
  void initState() {
    refreshController = RefreshController();
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback(
      (_) {
        ref
            .read(serviceProvider.notifier)
            .fetchServices(context: context, isRefresh: true);
      },
    );
  }

  @override
  Widget build(BuildContext context) {
    final state = ref.watch(serviceProvider);
    final notifier = ref.read(serviceProvider.notifier);
    return Directionality(
      textDirection:
          LocalStorage.getLangLtr() ? TextDirection.ltr : TextDirection.rtl,
      child: Padding(
        padding: const EdgeInsets.all(8.0),
        child: LayoutBuilder(builder: (context, constraints) {
          return SingleChildScrollView(
              child: Column(children: [
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.service),
                  style: Style.interMedium(size: 20),
                ),
                const Spacer(),
                CustomRefresher(
                    onTap: () {
                      notifier.fetchServices(context: context, isRefresh: true);
                    },
                    isLoading: state.isLoading),
                Padding(
                  padding: const EdgeInsets.all(12.0),
                  child: LoginButton(
                    bgColor: Style.black,
                    icon: Icon(
                      Remix.add_circle_line,
                      color: Style.white,
                      size: 24.r,
                    ),
                    onPressed: () {
                      showDialog(
                          context: context,
                          builder: (_) {
                            return Dialog(
                                backgroundColor: Style.transparent,
                                child: AddServicePage());
                          });
                    },
                    title: AppHelpers.getTranslation(TrKeys.addService),
                  ),
                )
              ],
            ),
            _topRow(constraints, ref),
            state.isLoading
                ? _loading()
                : Column(
                    children: [
                      ...state.services.mapIndexed((e, i) {
                        return _tableRow(
                          e,
                          constraints,
                          ref,
                          onTap: () {
                            ref
                                .read(addServiceMasterProvider.notifier)
                                .setDetails(
                                  e,
                                )
                                .then((v) {
                              if (context.mounted) {
                                showDialog(
                                    context: context,
                                    builder: (_) {
                                      return Dialog(
                                        backgroundColor: Style.transparent,
                                        child: UpdateServicePage(
                                          id: e.id ?? 0,
                                        ),
                                      );
                                    });
                              }
                            });
                          },
                          onDelete: () {
                            AppHelpers.showAlertDialog(context: context, child: DeleteDialog(success: (){
                              Navigator.pop(context);
                              ref
                                  .read(serviceMasterProvider.notifier)
                                  .deleteService(context, e.id)
                                  .then((t) {
                                if (context.mounted) {
                                  notifier.fetchServices(
                                      context: context, isRefresh: true);
                                }
                              });
                            }));
                          },
                        );
                      }),
                      16.verticalSpace,

                      if (state.hasMore)
                        HasMoreButton(
                          hasMore: state.hasMore,
                          onViewMore: () {
                            notifier.fetchServices(
                                context: context);
                          },
                        )
                    ],
                  ),
          ]));
        }),
      ),
    );
  }

  Widget _tableRow(
      ServiceData service, BoxConstraints constraints, WidgetRef ref,
      {required VoidCallback onTap, required VoidCallback onDelete}) {
    return Container(
      width: double.infinity,
      height: 60,
      decoration: BoxDecoration(
          color: Style.white,
          border: Border.fromBorderSide(BorderSide(color: Style.greyColor))),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.start,
        children: [
          12.horizontalSpace,
          CustomCheckbox(
            isActive:
                ref.watch(serviceProvider).selectProducts.contains(service.id),
            onTap: () {
              ref.read(serviceProvider.notifier).addSelectOrder(
                  id: service.id,
                  orderLength: ref.watch(serviceProvider).services.length);
            },
          ),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 17, child: Text("${service.id}")),

          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 5,
              child: Text(service.service?.translation?.title ?? "  -")),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 6,
              child: Text(AppHelpers.numberFormat(number:
                  service.price))),
          16.horizontalSpace,
          SizedBox(
            width: constraints.maxWidth / 7,
            child: Text("${service.interval}"),
          ),
          16.horizontalSpace,
          if (constraints.maxWidth > 1100)
            SizedBox(
              width: constraints.maxWidth / 5,
              child: Text(TimeService.dateFormatMDYHm(service.createdAt)),
            ),
          Spacer(),
          ServicePopUp(onEdit: onTap, onDelete: onDelete),
          16.horizontalSpace,
        ],
      ),
    );
  }

  Container _topRow(BoxConstraints constraints, WidgetRef ref) {
    return Container(
      width: double.infinity,
      height: 60,
      decoration: BoxDecoration(
        color: Style.white,
        borderRadius: BorderRadius.only(
          topLeft: Radius.circular(AppConstants.radius),
          topRight: Radius.circular(AppConstants.radius),
        ),
        border: Border.fromBorderSide(
          BorderSide(color: Style.greyColor),
        ),
      ),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.start,
        children: [
          16.horizontalSpace,
          CustomCheckbox(
            isActive: ref.watch(serviceProvider).isAllSelect,
            onTap: () {
              ref
                  .read(serviceProvider.notifier)
                  .allSelectOrder(ref.watch(serviceProvider).services);
            },
          ),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth/17,
              child: Text(
                AppHelpers.getTranslation(TrKeys.id),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 5,
              child: Text(
                AppHelpers.getTranslation(TrKeys.name),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 6,
              child: Text(
                AppHelpers.getTranslation(TrKeys.price),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 7,
              child: Text(
                AppHelpers.getTranslation(TrKeys.interval),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          if (constraints.maxWidth > 1100)
            SizedBox(
                width: constraints.maxWidth / 5,
                child: Text(
                  AppHelpers.getTranslation(TrKeys.createdDate),
                  style: Style.interMedium(),
                )),
          16.horizontalSpace
        ],
      ),
    );
  }

  ListView _loading() {
    return ListView.builder(
        itemCount: 10,
        shrinkWrap: true,
        itemBuilder: (context, index) {
          return Container(
            height: 60,
            width: double.infinity,
            decoration: BoxDecoration(
                color: Style.greyColor,
                border: Border.fromBorderSide(BorderSide(color: Style.white))),
          );
        });
  }
}
